/*
  Antenna Rotator Turn Indicator
  Yaesu G-450A version

  Glen Popiel - KW5GP

  Uses Adafruit NeoPixel Library

  Released under the GPLv3 license
*/

//#define debug // Uncomment this to enable the display of debug information on the Serial Monitor

#include <Adafruit_NeoPixel.h>  // Include the Adafruit Neopixel library

#define led_pin 6 // Define the Neopixel I/O pin

#define num_pixels 12 // Define the NeoPixel Strip/Ring size

#define ccw_pin 3 //define the counterclockwise rotation input pin
#define cw_pin 4 // define the clockwise rotation input pin
#define debounce 20 // Define the input debounce delay

// Setup the NeoPixel library
// Uncomment the following for standard RGB Neopixel/WS281x LED Strips
//Adafruit_NeoPixel pixels(num_pixels, led_pin, NEO_GRB + NEO_KHZ800);

// Uncomment the following for RGBW Neopixel/WS281x LED Strips (RGB w/White LED)
// If you look at the unlit LEDs, half of each LED is covered with yellow
Adafruit_NeoPixel pixels(num_pixels, led_pin, NEO_GRBW + NEO_KHZ800);

#define delay_time 500 // Time (in milliseconds) to pause between pixels

int current_pixel, last_pixel; // Pixel variables
bool cw_in, ccw_in, moving; // Rotation variables

void setup()
{
#ifdef debug
  Serial.begin(9600); // Start the serial port if debugging is enabled
  Serial.println("Starting");
#endif

  pinMode(cw_pin, INPUT_PULLUP);  // Set the pin mode for the cw and ccw inputs
  pinMode(ccw_pin, INPUT_PULLUP);

  pixels.begin(); // Initialize the NeoPixel display
  pixels.clear(); // Set all pixel colors to 'off'
  pixels.show();   // Send the updated pixel colors to the hardware
  moving = false; // Clear the moving flag
}

void loop()
{
  cw_in = digitalRead(cw_pin);  // Read the cw input
  ccw_in = digitalRead(ccw_pin);  // read the ccw input
  delay(debounce);

  /*
    The Yaesu G-450 inputs are different from the G-800/1000. The G-450 has an AC-powered rotator motor
    The only good place to get the sensing information is across the front panel switches
    These switches are connected in such as way as to give an unexpected set of signal outputs
    to the Arduino Board

    When idle both inputs are low
    When rotating Left (CCW) only the CCW input goes high
    When rotating right (CW) but inputs go high
  */

  if ((cw_in == HIGH))  // This goes high regardless of direction so we only need to check it
  {
#ifdef debug
    Serial.print("   CCW = ");
    Serial.print(ccw_in);
    Serial.print("   CW = ");
    Serial.println(cw_in);
    Serial.println("CW or CCW Switch pressed");
#endif

    moving = true;  // Set the rotation flag
    if (ccw_in == HIGH && cw_in == HIGH)  // If both inputs are high we're rotating clockwise (right)
    {
      // we're turning clockwise

#ifdef debug
      Serial.println("Turning Clockwise");
#endif
      turn_cw();  // Call the turn_cw function
    }

    if ( ccw_in == LOW && cw_in == HIGH)  // If ccw is low and cw is high we're rotating counterclockwise (left)
    {

      // We're turning counterclockwise

#ifdef debug
      Serial.println("Turning Counterclockwise");
#endif

      turn_ccw(); // Call the turn_ccw function
    }
  } else
  {
    if (moving) // Have we been moving and aren't anymore? If so, flash the brake lights
    {
      moving = false; // Turn off the rotating flag
    }
    //nothing happening- clear the LED's
    pixels.clear(); // Set all pixel colors to 'off'
    pixels.show();   // Send the updated pixel colors to the hardware
  }
}

void turn_cw()  // The turn_cw function
{
  int current_pixel = 0;  // Select the starting LED
  delay(debounce);
  while (digitalRead(ccw_pin) == HIGH && digitalRead(cw_pin == HIGH)) // Repeat the loop as long as we're rotating clockwise
  {
    pixels.setPixelColor(current_pixel, pixels.Color(0, 150, 0));    // For each pixel set LED color to green
    pixels.show();   // Send the updated pixel colors to the hardware.

    if (current_pixel >= num_pixels)  // If we've lit all the LED's
    {
      // clear all pixels
      pixels.clear(); // Set all pixel colors to 'off'
      pixels.show();   // Send the updated pixel colors to the hardware
      current_pixel = current_pixel - num_pixels; // Select the first LED
    } else
    {
      current_pixel = current_pixel + 1;  // Light another LED
    }
    delay(delay_time); // Pause before next pass through loop
  }
  brake();  // Call the brake() function when switch is released
}


void turn_ccw()  // The turn_ccw function
{
  int current_pixel = num_pixels - 1;  // Select the starting LED
  delay(debounce);
  while (digitalRead(ccw_pin) == LOW && digitalRead(cw_pin) == HIGH)
  {
    pixels.setPixelColor(current_pixel, pixels.Color(0, 150, 0));    // For each pixel set LED color to green
    pixels.show();   // Send the updated pixel colors to the hardware.

    if (current_pixel < 0)
    {
      // clear all pixels
      pixels.clear(); // Set all pixel colors to 'off'
      pixels.show();   // Send the updated pixel colors to the hardware
      current_pixel = num_pixels - 1; // Select the first LED
    } else
    {
      current_pixel = current_pixel - 1;  // Light another LED
    }
    delay(delay_time); // Pause before next pass through loop
  }
  brake();  // Call the brake() function
}

void brake()  // The brake() function
{
#ifdef debug
  Serial.println("Braking");
#endif

  pixels.fill(0, 0, num_pixels);  // Clear all the LEDs
  pixels.show();   // Send the updated pixel colors to the hardware.
  delay(delay_time);
  for (int i = 0; i < 5; i = i + 1)  // Flash all LEDs for 5 seconds for braking indicator
  {
    pixels.fill(pixels.Color(150, 0, 0), 0, num_pixels);  // Set the LED color to red
    pixels.show();   // Send the updated pixel colors to the hardware.
    delay(delay_time * 2);
    pixels.fill(0, 0, num_pixels);
    pixels.show();   // Send the updated pixel colors to the hardware.
    delay(delay_time * 2);
  }
}

